<?php
// administrator02.php with search functionality

require_once 'rb.php';

// --- CONFIGURATION ---
$db_file = 'mwalimu.db'; // Change this to your SQLite file or DSN
$dsn = "sqlite:$db_file";
R::setup($dsn);
R::freeze(false); // Allow schema changes

// --- HELPER: Escape HTML ---
function h($str) { return htmlspecialchars($str, ENT_QUOTES, 'UTF-8'); }

// --- ROUTE PARAMS ---
$table = $_GET['table'] ?? '';
$action = $_GET['action'] ?? 'list';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 10; // Records per page
$search = trim($_GET['search'] ?? '');

// --- Fetch tables ---
$tables = R::inspect();

// --- Validate table param ---
if ($table !== '' && !in_array($table, $tables)) {
    die('Invalid table.');
}

// --- Pagination helper ---
function paginate($total, $per_page, $page, $base_url) {
    $pages = max(1, ceil($total / $per_page));
    echo '<nav aria-label="Pagination"><ul class="pagination">';
    for ($i = 1; $i <= $pages; $i++) {
        $active = ($i == $page) ? 'aria-current="page"' : '';
        echo '<li><a href="' . h($base_url) . '&page=' . $i . "\" $active>$i</a></li>";
    }
    echo '</ul></nav>';
}

// --- Build search SQL and params ---
$where = '';
$params = [];
if ($search !== '' && $table !== '') {
    // Get columns and their types to filter text columns only
    $columns = R::getColumns($table);
    $textColumns = [];
    foreach ($columns as $col => $type) {
        // Simple check for text-like columns (SQLite types)
        if (stripos($type, 'char') !== false || stripos($type, 'text') !== false) {
            $textColumns[] = $col;
        }
    }
    if (count($textColumns) > 0) {
        $likeClauses = [];
        foreach ($textColumns as $col) {
            $likeClauses[] = "$col LIKE ?";
            $params[] = '%' . $search . '%';
        }
        $where = 'WHERE ' . implode(' OR ', $likeClauses);
    }
}

// --- Process POST requests ---
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($action === 'create' && $table) {
        $bean = R::dispense($table);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<b style="color:green">Record created.</b>';
        $action = 'list';
    }
    if ($action === 'update' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<b style="color:green">Record updated.</b>';
        $action = 'list';
    }
    if ($action === 'delete' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        R::trash($bean);
        $message = '<b style="color:green">Record deleted.</b>';
        $action = 'list';
    }
}

?><!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>Perplexity Adminer - Administrator02 with Search</title>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@picocss/pico@1.4.0/css/pico.min.css" />
<script src="https://cdn.jsdelivr.net/npm/prototype@1.7.3/dist/prototype.min.js"></script>
<style>
  body {
    display: flex;
    justify-content: center;
    align-items: flex-start;
    min-height: 100vh;
    padding: 2rem;
    background: #f5f7fa;
  }
  main.card {
    max-width: 960px;
    width: 100%;
    padding: 2rem;
    background: white;
    border-radius: 8px;
    box-shadow: 0 0 15px rgba(0,0,0,0.1);
  }
  table {
    width: 100%;
    border-collapse: collapse;
  }
  th, td {
    padding: 0.3rem 0.5rem;
    border: 1px solid #ccc;
    text-align: left;
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
  }
  nav.pagination ul {
    list-style: none;
    padding: 0;
    display: flex;
    gap: 0.5rem;
  }
  nav.pagination li a {
    padding: 0.3rem 0.6rem;
    border: 1px solid #ccc;
    border-radius: 4px;
    text-decoration: none;
  }
  nav.pagination li a[aria-current="page"] {
    background: #007bff;
    color: white;
    pointer-events: none;
  }
  nav > a {
    margin-right: 0.5rem;
  }
  form.search-form {
    margin-bottom: 1rem;
  }
</style>
</head>
<body>
<main class="card" role="main">
<header>
    <h1>Perplexity Adminer - Administrator02 with Search</h1>
</header>

<section>
    <h2>Tables</h2>
    <nav>
        <?php foreach ($tables as $t): ?>
            <a href="?table=<?=h($t)?>&action=list"><?=h($t)?></a>
        <?php endforeach; ?>
    </nav>
</section>

<section>
    <?php if ($message): ?>
        <p><?= $message ?></p>
    <?php endif; ?>

    <?php if ($table === ''): ?>
        <p>Select a table to manage.</p>

    <?php elseif ($action === 'list'): ?>
        <h2>Table: <?=h($table)?></h2>
        <nav>
            <a href="?table=<?=h($table)?>&action=create">Create New Record</a>
        </nav>

        <!-- Search form -->
        <form method="get" class="search-form" action="">
            <input type="hidden" name="table" value="<?=h($table)?>">
            <input type="hidden" name="action" value="list">
            <label for="search">Search:</label>
            <input type="search" id="search" name="search" value="<?=h($search)?>" placeholder="Search in text fields..." />
            <button type="submit">Filter</button>
            <?php if ($search !== ''): ?>
                <a href="?table=<?=h($table)?>&action=list">Clear</a>
            <?php endif; ?>
        </form>

        <br>
        <?php
        // Count total with search filter
        if ($where) {
            $total = R::count($table, $where, $params);
        } else {
            $total = R::count($table);
        }
        $offset = ($page - 1) * $per_page;

        // Fetch records with search filter
        if ($where) {
            $records = R::findAll($table, "$where LIMIT ? OFFSET ?", array_merge($params, [$per_page, $offset]));
        } else {
            $records = R::findAll($table, 'LIMIT ? OFFSET ?', [$per_page, $offset]);
        }

        if ($total === 0) {
            echo "<p>No records found.</p>";
        } else {
            echo '<table><thead><tr>';
            $first = reset($records);
            if ($first) {
                $cols = array_keys((array)$first->export());
                $cols_to_show = array_slice($cols, 0, 3);
                foreach ($cols_to_show as $col) {
                    echo '<th>' . h($col) . '</th>';
                }
                echo '<th>Actions</th>';
                echo '</tr></thead><tbody>';
                foreach ($records as $record) {
                    echo '<tr>';
                    foreach ($cols_to_show as $col) {
                        $val = $record->$col;
                        echo '<td title="' . h($val) . '">' . h($val) . '</td>';
                    }
                    echo '<td>';
                    echo '<a href="?table=' . h($table) . '&action=details&id=' . $record->id . '">Details</a> | ';
                    echo '<a href="?table=' . h($table) . '&action=update&id=' . $record->id . '">Edit</a> | ';
                    echo '<a href="?table=' . h($table) . '&action=delete&id=' . $record->id . '">Delete</a>';
                    echo '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
                // Build base url for pagination including search param if any
                $base_url = "?table=" . urlencode($table) . "&action=list";
                if ($search !== '') {
                    $base_url .= "&search=" . urlencode($search);
                }
                paginate($total, $per_page, $page, $base_url);
            }
        }
        ?>

    <?php elseif ($action === 'details' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
            echo "<h2>Details of Record #".h($id)." in ".h($table)."</h2>";
            echo '<table>';
            foreach ($bean as $col => $val) {
                echo '<tr><th>' . h($col) . '</th><td>' . nl2br(h($val)) . '</td></tr>';
            }
            echo '</table>';
            echo '<p><a href="?table=' . h($table) . '&action=list">Back to list</a></p>';
        }
        ?>

    <?php elseif ($action === 'create'): ?>
        <h2>Create New Record in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=create">
            <?php
            $columns = R::inspect($table);
            foreach ($columns as $col):
                if ($col === 'id') continue; // skip id
            ?>
                <label for="<?=h($col)?>"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" />
            <?php endforeach; ?>
            <button type="submit">Create</button>
            <a href="?table=<?=h($table)?>&action=list">Cancel</a>
        </form>

    <?php elseif ($action === 'update' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
        ?>
        <h2>Update Record #<?=h($id)?> in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=update&id=<?=h($id)?>">
            <input type="hidden" name="id" value="<?=h($id)?>">
            <?php
            foreach ($bean as $col => $val):
                if ($col === 'id') continue;
            ?>
                <label for="<?=h($col)?>"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" value="<?=h($val)?>" />
            <?php endforeach; ?>
            <button type="submit">Update</button>
            <a href="?table=<?=h($table)?>&action=list">Cancel</a>
        </form>
        <?php } ?>

    <?php elseif ($action === 'delete' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
        ?>
        <h2>Delete Record #<?=h($id)?> in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=delete&id=<?=h($id)?>">
            <input type="hidden" name="id" value="<?=h($id)?>">
            <p>Are you sure you want to delete this record?</p>
            <button type="submit">Delete</button>
            <a href="?table=<?=h($table)?>&action=list">Cancel</a>
        </form>
        <?php } ?>

    <?php else: ?>
        <p>Unknown action.</p>
    <?php endif; ?>
</section>

<footer>
    <p style="text-align:center; font-size:small; color:#666;">Perplexity Adminer &mdash; Powered by RedBeanPHP & Pico CSS</p>
</footer>
</main>
</body>
</html>
