<?php
// public_school_add.php (partial template)
// Assumes RedBeanPHP is loaded and R::setup() called in parent

$error = "";
$success = "";

// --- TABLE CREATION LOGIC ---
try {
    $exists = R::getCell("SELECT name FROM sqlite_master WHERE type='table' AND name='public_school'");

    if (!$exists) {
        R::exec("CREATE TABLE public_school (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            status TEXT NOT NULL,
            province TEXT NOT NULL,
            district TEXT NOT NULL,
            division TEXT NOT NULL,
            location TEXT NOT NULL,
            constituency TEXT NOT NULL,
            county TEXT NOT NULL,
            email TEXT,
            latitude REAL,
            longitude REAL,
            created_at DATETIME
        )");
    }
} catch (Exception $e) {
    $error = 'Table creation error: ' . $e->getMessage();
}

// --- HANDLE FORM SUBMISSION ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $data = [
            'name' => $_POST['name'] ?? '',
            'status' => $_POST['status'] ?? '',
            'province' => $_POST['province'] ?? '',
            'district' => $_POST['district'] ?? '',
            'division' => $_POST['division'] ?? '',
            'location' => $_POST['location'] ?? '',
            'constituency' => $_POST['constituency'] ?? '',
            'county' => $_POST['county'] ?? '',
            'email' => $_POST['email'] ?? '',
            'latitude' => $_POST['latitude'] ?? null,
            'longitude' => $_POST['longitude'] ?? null,
            'created_at' => date('Y-m-d H:i:s')
        ];

        // Validate required fields
        $required = ['name', 'status', 'province', 'district', 'division', 'location', 'constituency', 'county'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                throw new Exception("Field " . ucfirst(str_replace('_', ' ', $field)) . " is required");
            }
        }

        // Validate latitude and longitude if provided
        if (!empty($data['latitude']) && !is_numeric($data['latitude'])) {
            throw new Exception("Latitude must be a number");
        }
        if (!empty($data['longitude']) && !is_numeric($data['longitude'])) {
            throw new Exception("Longitude must be a number");
        }

        // Insert into database using parameterized query
        $columns = implode(', ', array_keys($data));
        $placeholders = ':' . implode(', :', array_keys($data));
        $sql = "INSERT INTO public_school ($columns) VALUES ($placeholders)";

        $pdo = R::getDatabaseAdapter()->getDatabase();
        $stmt = $pdo->prepare($sql);

        foreach ($data as $key => $value) {
            $param = ($value === null) ? PDO::PARAM_NULL : PDO::PARAM_STR;
            $stmt->bindValue(':' . $key, $value, $param);
        }

        if ($stmt->execute()) {
            $success = "School added successfully!";
            $_POST = []; // Clear form inputs
        } else {
            throw new Exception("Failed to save school record.");
        }
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    }
}
?>

<style>
    body { font-family: Arial, sans-serif; margin: 2em; }
    .form-container { max-width: 800px; margin: auto; }
    .form-group { margin-bottom: 1em; }
    .form-row { display: flex; gap: 1em; flex-wrap: wrap; }
    .form-row .form-group { flex: 1 1 45%; min-width: 250px; }
    label { display: block; margin-bottom: 0.5em; font-weight: bold; }
    input, select {
        width: 100%;
        padding: 8px;
        margin-bottom: 0.5em;
        border: 1px solid #ccc;
        border-radius: 4px;
        box-sizing: border-box;
    }
    .error { color: red; margin: 1em 0; padding: 1em; border: 1px solid red; }
    .success { color: green; margin: 1em 0; padding: 1em; border: 1px solid green; }
    .required::after { content: "*"; color: red; margin-left: 4px; }
    .form-actions { margin-top: 1.5em; }
    button { padding: 8px 20px; cursor: pointer; }
</style>

<div class="form-container">
    <h2>Add New Public School</h2>

    <?php if ($error): ?>
        <div class="error"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <form method="post" novalidate>
        <div class="form-row">
            <div class="form-group">
                <label for="name" class="required">School Name</label>
                <input type="text" id="name" name="name" value="<?= htmlspecialchars($_POST['name'] ?? '') ?>" required>
            </div>

            <div class="form-group">
                <label for="status" class="required">Status</label>
                <select id="status" name="status" required>
                    <option value="">Select Status</option>
                    <option value="Public" <?= (($_POST['status'] ?? '') === 'Public') ? 'selected' : '' ?>>Public</option>
                    <option value="Private" <?= (($_POST['status'] ?? '') === 'Private') ? 'selected' : '' ?>>Private</option>
                    <option value="Charter" <?= (($_POST['status'] ?? '') === 'Charter') ? 'selected' : '' ?>>Charter</option>
                </select>
            </div>
        </div>

        <div class="form-row">
            <div class="form-group">
                <label for="province" class="required">Province</label>
                <input type="text" id="province" name="province" value="<?= htmlspecialchars($_POST['province'] ?? '') ?>" required>
            </div>

            <div class="form-group">
                <label for="county" class="required">County</label>
                <input type="text" id="county" name="county" value="<?= htmlspecialchars($_POST['county'] ?? '') ?>" required>
            </div>
        </div>

        <div class="form-row">
            <div class="form-group">
                <label for="district" class="required">District</label>
                <input type="text" id="district" name="district" value="<?= htmlspecialchars($_POST['district'] ?? '') ?>" required>
            </div>

            <div class="form-group">
                <label for="division" class="required">Division</label>
                <input type="text" id="division" name="division" value="<?= htmlspecialchars($_POST['division'] ?? '') ?>" required>
            </div>
        </div>

        <div class="form-row">
            <div class="form-group">
                <label for="location" class="required">Location</label>
                <input type="text" id="location" name="location" value="<?= htmlspecialchars($_POST['location'] ?? '') ?>" required>
            </div>

            <div class="form-group">
                <label for="constituency" class="required">Constituency</label>
                <input type="text" id="constituency" name="constituency" value="<?= htmlspecialchars($_POST['constituency'] ?? '') ?>" required>
            </div>
        </div>

        <div class="form-row">
            <div class="form-group">
                <label for="email">Email Address</label>
                <input type="email" id="email" name="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">
            </div>

            <div class="form-group">
                <label for="latitude">Latitude</label>
                <input type="number" step="any" id="latitude" name="latitude" value="<?= htmlspecialchars($_POST['latitude'] ?? '') ?>" placeholder="e.g., -1.2921">
            </div>

            <div class="form-group">
                <label for="longitude">Longitude</label>
                <input type="number" step="any" id="longitude" name="longitude" value="<?= htmlspecialchars($_POST['longitude'] ?? '') ?>" placeholder="e.g., 36.8219">
            </div>
        </div>

        <div class="form-actions">
            <button type="submit">Add School</button>
            <a href="public_school_search.php" style="margin-left: 1em;">Back to List</a>
        </div>
    </form>
</div>
