<?php
require 'rb.php'; // RedBeanPHP

// Connect to SQLite
R::setup('sqlite:mwalimu.db');

// CONFIG
$resetTokenExpiry = 60 * 60; // 1 hour

// Helper: Set and get cookie messages
function set_flash($msg, $type = 'msg') {
    setcookie($type, $msg, time() + 60, "/");
}
function get_flash($type = 'msg') {
    if (isset($_COOKIE[$type])) {
        $msg = $_COOKIE[$type];
        setcookie($type, '', time() - 3600, "/");
        return $msg;
    }
    return '';
}

// STEP 1: REQUEST PASSWORD RESET
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['request_reset'])) {
    $email = trim($_POST['email']);
    $user = null;
    $userType = null;

    // Try to find user in teacher table
    $user = R::findOne('teacher', 'email = ?', [$email]);
    if ($user) $userType = 'teacher';

    // If not found, try school table (use "email" column)
    if (!$user) {
        $user = R::findOne('school', 'email = ?', [$email]);
        if ($user) $userType = 'school';
    }

    // Always show generic message
    set_flash("If this email exists in our records, a reset link has been sent.");

    if ($user) {
        // Generate token and expiry
        $token = bin2hex(random_bytes(32));
        $expiry = time() + $resetTokenExpiry;

        // Store token and expiry
        $user->reset_token = $token;
        $user->reset_expiry = $expiry;
        R::store($user);

        // Send email (replace with your mail logic)
        $resetLink = "http://{$_SERVER['HTTP_HOST']}{$_SERVER['PHP_SELF']}?token=$token&type=$userType";
        $subject = "Password Reset Request";
        $body = "Hello,\n\nClick the link below to reset your password. This link will expire in 1 hour.\n\n$resetLink\n\nIf you did not request this, please ignore this email.";
        $headers = "From: no-reply@yourdomain.com\r\n";
        @mail($email, $subject, $body, $headers);
    }
    header("Location: ".$_SERVER['PHP_SELF']);
    exit;
}

// STEP 2: SHOW RESET FORM IF TOKEN PRESENT
$showResetForm = false;
$error = '';
$success = '';

if (isset($_GET['token']) && isset($_GET['type'])) {
    $token = $_GET['token'];
    $type = $_GET['type'];
    $user = R::findOne($type, 'reset_token = ?', [$token]);

    if ($user && $user->reset_expiry >= time()) {
        $showResetForm = true;
    } else {
        $error = "Invalid or expired reset link.";
    }

    // Handle password reset submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_password'])) {
        $newPassword = $_POST['password'];
        $confirmPassword = $_POST['confirm_password'];
        if ($newPassword !== $confirmPassword) {
            $error = "Passwords do not match.";
            $showResetForm = true;
        } elseif (strlen($newPassword) < 6) {
            $error = "Password must be at least 6 characters.";
            $showResetForm = true;
        } else {
            // Update password (hash it!)
            $user->password = password_hash($newPassword, PASSWORD_DEFAULT);
            $user->reset_token = null;
            $user->reset_expiry = null;
            R::store($user);
            set_flash("Password reset successful! You can now <a href='login.php'>log in</a>.", 'msg');
            header("Location: ".$_SERVER['PHP_SELF']);
            exit;
        }
    }
}

// Get and clear flash messages
$message = get_flash('msg');
$error = $error ?: get_flash('error');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Reset Password</title>
    <link rel="stylesheet" href="https://unpkg.com/@picocss/pico@2.0.6/css/pico.min.css">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        main { max-width: 400px; margin: 3rem auto; }
        .msg { color: #157347; }
        .error { color: #b02a37; }
    </style>
</head>
<body>
<main>
    <h2>Reset Password</h2>
    <?php if ($message): ?>
        <article class="msg"><?php echo $message; ?></article>
    <?php endif; ?>
    <?php if ($error): ?>
        <article class="error"><?php echo $error; ?></article>
    <?php endif; ?>

    <!-- Step 1: Request Reset Form -->
    <?php if (!isset($_GET['token'])): ?>
        <form method="post" autocomplete="off">
            <label for="email">Enter your registered email:</label>
            <input type="email" name="email" id="email" required autocomplete="off">
            <button type="submit" name="request_reset">Send Reset Link</button>
        </form>
    <?php endif; ?>

    <!-- Step 2: New Password Form -->
    <?php if ($showResetForm): ?>
        <form method="post" autocomplete="off">
            <label for="password">New Password:</label>
            <input type="password" name="password" id="password" required minlength="6" autocomplete="new-password">
            <label for="confirm_password">Confirm New Password:</label>
            <input type="password" name="confirm_password" id="confirm_password" required autocomplete="new-password">
            <button type="submit" name="reset_password">Reset Password</button>
        </form>
    <?php endif; ?>
</main>
</body>
</html>
